module net.BurtonRadons.dig.common.color;

//private import std.math;

/** An RGBA color. */
struct Color
{
    ubyte r = 0; /**< The red component, from 0 (black) to 255 (red). */
    ubyte g = 0; /**< The green component, from 0 (black) to 255 (green). */
    ubyte b = 0; /**< The blue component, from 0 (black) to 255 (blue). */
    ubyte a = 255; /**< The opacity component, from 0 (transparent) to 255 (opaque). */

    /** The color black. */
    const Color Black = { 0, 0, 0, 255 };

    /** The color white. */
    const Color White = { 255, 255, 255, 255 };

    /** A transparent color. */
    const Color Trans = { 0, 0, 0, 0 };

    /** Blend two colors together using the second color's opacity. */
    Color blend (Color o)
    {
        ubyte t = o.a;
        ubyte u = 255 - t;

        return AColor ((r * u + o.r * t) / 255,
                       (g * u + o.g * t) / 255,
                       (b * u + o.b * t) / 255,
                       255 - (255 - a) * (255 - t) / 255);
    }

    /** Blend two colors together using the given opacity. */
    Color blend (Color o, ubyte t)
    {
        ubyte u = 255 - t;

        return AColor ((r * u + o.r * t) / 255,
                       (g * u + o.g * t) / 255,
                       (b * u + o.b * t) / 255,
                       (a * u + o.a * t) / 255);
    }

    /** Blend two colors together with a saturated blend opacity; t should be between 0 and 255. */
    Color sblend (Color o, float t)
    {
        return blend (o, t < 0 ? 0 : t > 255 ? 255 : t);
    }

    /** Blend four colors together; they're positioned as in a 2x2 matrix.
      * The left and right colors are blended using a, then those top and
      * bottom colors are blended using b.
      */

    Color blend2 (Color yx, Color xy, Color yy, ubyte a, ubyte b)
    {
        Color xx = *this;
        Color x = xx.blend (yx, a);
        Color y = xy.blend (yy, a);

        return x.blend (y, b);
    }

    /** Convert the color to HSV; each component is from 0 to 1.0. */

    void toHSV (out float h, out float s, out float v)
    {
        float hr = r / 255.0, hg = g / 255.0, hb = b / 255.0;
        float t;

        v = fmax (hr, hg, hb);
        t = fmin (hr, hg, hb);
        if (v < 0.0001)
            s = 0;
        else
            s = (v - t) / v;

        if (s == 0)
            h = float.nan;
        else
        {
            float cr = (v - hr) / (v - t);
            float cg = (v - hg) / (v - t);
            float cb = (v - hb) / (v - t);

            if (hr == v)
                h = cb - cg;
            else if (hg == v)
                h = 2 + cr - cb;
            else if (hb == v)
                h = 4 + cg - cr;
            h /= 6.0;
            if (h < 0)
                h += 1;
        }
    }

    /** Get a color from HSV. */

    static Color fromHSV (float h, float s, float v) { return fromHSV (h, s, v, 255); }

    /** Get a color from HSV, with an explicit opacity. */

    static Color fromHSV (float h, float s, float v, ubyte a)
    {
        Color c;

        if (s < 0.001 || std.math.isnan (h))
            return SColor (v * 255, v * 255, v * 255, a);
        else
        {
            float f, m, n, k;
            int i;

            h = (h % 1.0) * 6;
            i = std.math.floor (h);
            f = h - i;
            m = v * (1 - s);
            n = v * (1 - s * f);
            k = v * (1 - s * (1 - f));

            switch (i)
            {
                case 0: return SColor (v * 255, k * 255, m * 255, a);
                case 1: return SColor (n * 255, v * 255, m * 255, a);
                case 2: return SColor (m * 255, v * 255, k * 255, a);
                case 3: return SColor (m * 255, n * 255, v * 255, a);
                case 4: return SColor (k * 255, m * 255, v * 255, a);
                case 5: return SColor (v * 255, m * 255, n * 255, a);
            }
        }
    }

    /** Store into a three-cell ubyte array. */
    void storeUByte3 (ubyte *p)
    {
        p [0] = r;
        p [1] = g;
        p [2] = b;
    }

    /** Get the color's intensity. */
    ubyte intensity ()
    {
        return (r * 2125 + g * 7154 + b * 721) / 10000;
    }

    /** Return the inverse of the color, or a diametrically opposed color when a component nears 128. */
    Color inverse ()
    {
        Color result;
        int side;

        side = r < 128 ? 255 : 0;
        result.r = (iabs (r - side) / 2 > iabs (r - ~r)) ? side : ~r;

        side = g < 128 ? 255 : 0;
        result.g = (iabs (g - side) / 2 > iabs (g - ~g)) ? side : ~g;

        side = b < 128 ? 255 : 0;
        result.b = (iabs (b - side) / 2 > iabs (b - ~b)) ? side : ~b;

        return result;
    }

    /** Get a string representation. */
    char[] toString ()
    {
        if (a == 255)
            return fmt ("Color (%d, %d, %d)", r, g, b);
        return fmt ("Color (%d, %d, %d, %d)", r, g, b, a);
    }

    /** Scale all parameters, where 0 results in black and 1 results in no change.   Opacity is passed unmolested. */
    Color opMul (float scale)  /* operator overloading for  comparison */
    {
        return SColor (r * scale, g * scale, b * scale, a);
    }

    /** Multiply and assign. */
    Color opMulAssign (float scale)  /* operator overloading for  comparison */
    {
        return *this = *this * scale;
    }

    /** Add two colors together.  Opacity is unmolested. */
    Color opAdd (Color other)   /* operator overloading for  comparison */
    {
        return SColor (r + other.r, g + other.g, b + other.g, a);
    }

    /** Add and assign. */
    Color opAddAssign (Color other)  /* operator overloading for  comparison */
    {
        return *this = *this + other;
    }
}

/** Construct a color, using 255 for opacity.
  *
  * @relates Color
  */

Color AColor (ubyte r, ubyte g, ubyte b) { Color c; c.r = r; c.g = g; c.b = b; return c; }

/** Construct a color with explicit opacity.
  *
  * @relates Color
  */

Color AColor (ubyte r, ubyte g, ubyte b, ubyte a) { Color c; c.r = r; c.g = g; c.b = b; c.a = a; return c; }

/** Construct a color, saturating each attribute to be in range.
  *
  * @relates Color
  */

Color SColor (float r, float g, float b)
{
    return AColor (r < 0 ? 0 : r > 255 ? 255 : r,
                   g < 0 ? 0 : g > 255 ? 255 : g,
                   b < 0 ? 0 : b > 255 ? 255 : b);
}

/** Construct a color, saturating each attribute to be in range.
  *
  * @relates Color
  */

Color SColor (float r, float g, float b, float a)
{
    return AColor (r < 0 ? 0 : r > 255 ? 255 : r,
                   g < 0 ? 0 : g > 255 ? 255 : g,
                   b < 0 ? 0 : b > 255 ? 255 : b,
                   a < 0 ? 0 : a > 255 ? 255 : a);
}

/** Construct a color with 0-1 range on each component.
  * @relates Color
  */
Color SColor1 (float r, float g, float b)
{
    return SColor (r * 255, g * 255, b * 255);
}

/** Construct a color with 0-1 range on each component.
  * @relates Color
  */
Color SColor1 (float r, float g, float b, float a)
{
    return SColor (r * 255, g * 255, b * 255, a * 255);
}
